library(Rsamtools)
library(GenomicRanges)
library(readr)
library(gplots)
library(viridis)
library(scales)

##call functions
source('/data/functions/create_typhoon_plot_functions.R')
source("/data/functions/misc_functions.R")

file_name = c('wt_pulse','wt_c10','wt_c15','wt_c20','wt_c40',
              'cac_pulse','cac_c10','cac_c15','cac_c20','cac_c40')

number_of_files=length(file_name)

###
# figure 1B, typhoon plot
###
chr= 'chrI'
new_start = 61500
new_end= 64000

data = vector("list")
read=vector("list")

for(f in 1:number_of_files){
  
  matx.m <- matrix(0, nrow=250, ncol=new_start - new_start + 1)

  file_name.bam = paste("~/data/bam_bai/",file_name[f],".bam", sep='')
  file_name.bam.bai = paste("~/data/bam_bai/",file_name[f],".bam.bai",sep='')
  
  #make into a GRanges object
  
  chr.gr <- GRanges(seqnames= chr, ranges = IRanges(start = new_start , end = new_end ))
  
  p <- ScanBamParam(what = c("rname", "strand", "pos", "isize"),which = chr.gr)
  
  A_reads.l <- scanBam(file = file_name.bam,
                       index = file_name.bam.bai,
                       param = p)
  
  A_reads.gr <- GRanges(seqnames = A_reads.l[[1]]$rname,
                        ranges = IRanges(start = A_reads.l[[1]]$pos,
                                         width = A_reads.l[[1]]$isize))
  
  A_reads.gr$midpoint <- mid(ranges(A_reads.gr))
  A_reads.gr <- A_reads.gr[which(A_reads.gr$midpoint <= new_end & A_reads.gr$midpoint >= new_start) ]
  A_reads.gr$relative_pos <- A_reads.gr$midpoint - new_start
  
  #save position and length of each read
  data[[f]] <- as.matrix(cbind(A_reads.gr$relative_pos, width(A_reads.gr)) )
  colnames(data[[f]]) <- c("x","y")
  
} #end file for

#plotting
scr.m = matrix(c(0.03, 0.5, 0.95, 1,
                 0.5, 0.97, 0.95, 1, #gene labels
                 
                 0.03, 0.5, 0.84, 0.95,
                 0.03, 0.5, 0.73, 0.84,
                 0.03, 0.5, 0.62, 0.73,
                 0.03, 0.5, 0.51, 0.62,
                 0.03, 0.5, 0.4, 0.51, #WT dot plots
                 
                 0.5, 0.97, 0.84, 0.95,
                 0.5, 0.97, 0.73, 0.84,
                 0.5, 0.97, 0.62, 0.73,
                 0.5, 0.97, 0.51, 0.62,
                 0.5, 0.97, 0.4, 0.51 #cac dot plots
                 
),
ncol = 4, byrow = T
)


file.name <- "/data/figure/figure1/figure1_typhoon.png"
png(file = file.name, width = 20, height = 18, units = "in", res = 200, bg = "white", type = "cairo-png" )
par(cex=1.7)

close.screen(all.screens = T)

split.screen(scr.m)

par(oma = c(0, 3, 0, 1)) 

for(j in 1:2){
  screen(j)
  par(mar = c(0,0.5,0,0.5), cex=1.5)  #margins order :bottom, left, top, right
  make_gene_schematic(chr, new_start, new_end, proteinCoding = F)#y_low = 0, y_high = 1,
}

# 
for (j in c(1:10)){
  screen((j+2))
  par(mar = c(0.5,0.5,0.5,0.5), cex=0.8, cex.main=0.7)  #margins order :bottom, left, top, right #cex.main is title magnif. relative to cex
  
  dcolor=densColsDM(data[[j]][,1],data[[j]][,2],nbin=c(1028,1028),bandwidth=c(36,16),transformation = function(x) x^0.3, #initially had decided on 0.5
                    colramp = colorRampPalette(blues9), factor=1)
 
  plot(c(-1,-1),c(-1,-1),main='', ylab="",xlab="",xaxt='n',yaxt="n",xaxs="i",xlim=c(new_start,new_end),ylim=c(20,250))
  abline(v =63000-10,lwd=18,col=alpha('darkolivegreen3',0.3)) #fast nuc
  abline(v =63500-5, lwd = 18, col=alpha('lightpink3',0.3)) #slow nuc
  points((data[[j]][,1]+new_start), y=data[[j]][,2],col=dcolor,cex=0.35,pch=20)
  legend('topright',as.character(nrow(data[[j]])),cex = 2,bty = "n")

  
  if(j %in% 1:5){
    axis(side=2, tick=T,at=c(50,150,250),cex.axis = 1.9)
  }
  
  if(j == 5 | j ==10){
    axis(side=1, at=seq(61500, 63500, 500), labels = TRUE,cex.axis=1.9)
  }
  
  
  box(which="plot")
  
} 

mtext('Position on ChrI (bp)', side = 1, outer = T, cex=2,line=-19,at = 0.5)
mtext('Fragment length (bp)', side = 2, outer = TRUE, cex=2, line=1,at = 0.7)

close.screen(all.screens = T)

dev.off()

###
#figure 1C, ACF boxplot 
###
#generated in acf_score.R
cl <- viridis(100, alpha = 1, begin = 0, end = 1, option = "D") #set color
wt_col = cl[55]
cac_col = 'orange'

acf.df = read.csv('~/data/acf_score.csv') 

acf.df = subset(acf.df,wt_c40 > median(acf.df$wt_c40)) #select genes with well positioned nucleosome arrays

n = nrow(acf.df)

#reformat data
acf_reorg = data.frame('acf'=c(acf.df$wt_pulse,acf.df$wt_c10,acf.df$wt_c15,acf.df$wt_c20,acf.df$wt_c40,
                               acf.df$cac_pulse,acf.df$cac_c10,acf.df$cac_c15,acf.df$cac_c20,acf.df$cac_c40),
                       'time'=rep(c(rep('Pulse',n),rep("10'",n),rep("15'",n),rep("20'",n),rep("40'",n)),2),
                       'cell'=c(rep('WT',5 * n),rep('cac1',5 * n)))
acf_reorg$time = factor(acf_reorg$time,levels = c('Pulse',"10'","15'","20'","40'"))
acf_reorg$cell = factor(acf_reorg$cell,levels = c('WT','cac1'))


#plot
ggplot(acf_reorg,aes(x=time,y=acf,fill=cell))+
  geom_boxplot(outlier.shape = NA,notch=T)+ylab('ACF')+xlab('')+ylim(-0.5,0.8)+theme_bw()+scale_fill_manual(name='',labels=c("WT",expression(paste(italic(cac1),Delta))),values = c(wt_col,cac_col))+
  theme(panel.grid.major = element_blank(),
        panel.grid.minor = element_blank(),
        panel.background = element_rect(colour = "black", size=0.5),
        axis.text  = element_text(size=17,face='bold'),
        axis.title = element_text(size=18),
        legend.text=element_text(size=18))
ggsave("~/data/figure/figure1/figure1_acf_boxplot.png", width =7, height =4.7, units = "in",dpi=400)


###
#figure 1 D, nuc occupancy box plot 
###
#generated with nuc_master.R
nuc_master  = read.csv('~/data/nuc_master.csv')

nuc_master_sub = subset(nuc_master,cac_c40 > quantile(nuc_master$cac_c40,0.25) & wt_c40 > quantile(nuc_master$wt_c40,0.25) ) #high-confidence nuc

n = nrow(nuc_master_sub)

nuc_reorg = data.frame('occupancy'=c(nuc_master_sub$wt_p,nuc_master_sub$wt_c10,nuc_master_sub$wt_c15,nuc_master_sub$wt_c20,nuc_master_sub$wt_c40,
                                     nuc_master_sub$cac_p,nuc_master_sub$cac_c10,nuc_master_sub$cac_c15,nuc_master_sub$cac_c20,nuc_master_sub$cac_c40),
                       'time'=rep(c(rep('Pulse',n),rep("10'",n),rep("15'",n),rep("20'",n),rep("40'",n)),2),
                       'cell'=c(rep('WT',5 * n),rep('cac1',5 * n)))
nuc_reorg$time = factor(nuc_reorg$time,levels = c('Pulse',"10'","15'","20'","40'"))
nuc_reorg$cell = factor(nuc_reorg$cell,levels = c('WT','cac1'))

#plot
ggplot(nuc_reorg,aes(x=time,y=occupancy,fill=cell))+
  geom_boxplot(outlier.shape = NA,notch=T)+ylab('Nucleosome occupancy')+xlab('')+ylim(0,500)+
  theme_bw()+ scale_fill_manual(name='',labels=c("WT",expression(paste(italic(cac1),Delta))),values = c(wt_col,cac_col))+
  theme(panel.grid.major = element_blank(), 
        panel.grid.minor = element_blank(),
        panel.background = element_rect(colour = "black", size=0.5),
        axis.text  = element_text(size=17,face='bold'),
        axis.title = element_text(size=18),
        legend.text=element_text(size=18))
ggsave("~/cac/final/figure/figure1/figure1_nuc_occu_boxplot.png", width =7, height =4.7, units = "in",dpi=400)

###
#supplemental 1, occupancy for highlighted slow/fast nucleosomes in cac
###
#fast nuc exp (green) is the 354th in nuc_master_sub  
#slow nuc exp (fast) is the 357th in nuc_master_sub

png(file = '~/data/figure/supple1_nuc_occupancy_single.png', width = 4, height = 3, units = "in", res = 300, bg = "white", type = "cairo-png" )
par(mar = c(4,4,2,1), cex=0.8, cex.main=0.7) 
plot(c(1,2,3,4,5),c(114,227,242,233,235),ylim=c(50,250),ylab='Occupancy',xaxt = 'n',col='forestgreen',pch=20,cex=2,xlab='')
points(c(1,2,3,4,5),c(91,96,87,145,171),col='tomato3',pch=20,cex=2)
axis(side = 1, at = c(1,2,3,4,5),labels = c('Pulse',"10'","15'","20'","40'"))
legend('bottomright',c('Nuc at 63,500 bp','Nuc at 63,000 bp'),col=c('tomato3','forestgreen'),pch=c(20,20),bty="n")
dev.off()




